<?php
/**
 * Plugin Name:       Direct Checkout Link Generator by 24siteshop
 * Description:       Generate one-click direct checkout links for WooCommerce products. Includes options to enforce single-product carts and hide cart/checkout notices.
 * Version:           1.0.1
 * Author:            24siteshop
 * Requires at least: 5.8
 * Requires PHP:      7.4
 * License:           GPL-2.0+
 * License URI:       http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain:       dclg-24siteshop
 */

if ( ! defined( 'ABSPATH' ) ) { exit; }

// Ensure WooCommerce is active
add_action( 'plugins_loaded', function () {
	if ( ! class_exists( 'WooCommerce' ) ) {
		add_action( 'admin_notices', function () {
			printf( '<div class="notice notice-error"><p>%s</p></div>', esc_html__( 'Direct Checkout Link Generator by 24siteshop requires WooCommerce to be active.', 'dclg-24siteshop' ) );
		});
	}
});

class DCLG_24SiteShop {
	const OPT_GROUP          = 'dclg_24siteshop_options_group';
	const OPT_KEY            = 'dclg_24siteshop_options';

	// Defaults
	private static function defaults() {
		return [
			'product_id'       => '',
			'enforce_single'   => 0, // 0 = allow multi-product carts; 1 = enforce single-product cart
			'suppress_notices' => 0,
		];
	}

	public static function init() {
		// Settings
		add_action( 'admin_menu', [ __CLASS__, 'register_menu' ] );
		add_action( 'admin_init', [ __CLASS__, 'register_settings' ] );

		// Enforce single-product cart if enabled
		add_filter( 'woocommerce_add_to_cart_validation', [ __CLASS__, 'maybe_enforce_single_product_cart' ], 10, 3 );

		// Suppress notices on cart/checkout if enabled
		add_action( 'template_redirect', [ __CLASS__, 'maybe_clear_cart_checkout_notices' ], 1 );
		add_filter( 'wc_add_to_cart_message_html', [ __CLASS__, 'maybe_suppress_add_to_cart_message' ], 20, 2 );

		// Settings link on Plugins screen
		add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), [ __CLASS__, 'settings_link' ] );
	}

	public static function settings_link( $links ) {
		$url = admin_url( 'admin.php?page=dclg-24siteshop' );
		$links[] = '<a href="' . esc_url( $url ) . '">' . esc_html__( 'Settings', 'dclg-24siteshop' ) . '</a>';
		return $links;
	}

	public static function register_menu() {
		add_menu_page(
			__( 'Direct Checkout Link', 'dclg-24siteshop' ),
			__( 'Direct Checkout Link', 'dclg-24siteshop' ),
			'manage_options',
			'dclg-24siteshop',
			[ __CLASS__, 'render_admin_page' ],
			'dashicons-cart',
			58
		);
	}

	public static function register_settings() {
		register_setting( self::OPT_GROUP, self::OPT_KEY, [ __CLASS__, 'sanitize_options' ] );
	}

	public static function sanitize_options( $input ) {
		$defaults = self::defaults();
		$output   = is_array( $input ) ? $input : [];

		$output['product_id']       = isset( $input['product_id'] ) ? sanitize_text_field( $input['product_id'] ) : $defaults['product_id'];
		$output['enforce_single']   = isset( $input['enforce_single'] ) ? 1 : 0;
		$output['suppress_notices'] = isset( $input['suppress_notices'] ) ? 1 : 0;

		return $output;
	}

	private static function get_options() {
		$opts = get_option( self::OPT_KEY, [] );
		return wp_parse_args( $opts, self::defaults() );
	}

	private static function build_direct_checkout_url( $product_id ) {
		if ( empty( $product_id ) ) {
			return '';
		}
		$checkout_url = wc_get_checkout_url();
		$args = [ 'add-to-cart' => $product_id ];
		return esc_url( add_query_arg( $args, $checkout_url ) );
	}

	public static function render_admin_page() {
		if ( ! current_user_can( 'manage_options' ) ) return;
		$opts = self::get_options();
		$generated = self::build_direct_checkout_url( $opts['product_id'] );
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'Direct Checkout Link Generator by 24siteshop', 'dclg-24siteshop' ); ?></h1>
			<p><?php esc_html_e( 'Generate a direct checkout link for any WooCommerce product. Optionally enforce a single-product cart and remove cart/checkout notices.', 'dclg-24siteshop' ); ?></p>

			<form method="post" action="options.php">
				<?php settings_fields( self::OPT_GROUP ); ?>
				<?php $option_name = self::OPT_KEY; ?>

				<table class="form-table" role="presentation">
					<tr>
						<th scope="row"><label for="dclg_product_id"><?php esc_html_e( 'Product ID', 'dclg-24siteshop' ); ?></label></th>
						<td>
							<input type="text" id="dclg_product_id" name="<?php echo esc_attr( $option_name ); ?>[product_id]" value="<?php echo esc_attr( $opts['product_id'] ); ?>" class="regular-text" placeholder="e.g. 123" />
							<p class="description"><?php esc_html_e( 'Enter the WooCommerce product ID, then click Save Changes. Use the button below to generate the link.', 'dclg-24siteshop' ); ?></p>
						</td>
					</tr>

					<tr>
						<th scope="row"><?php esc_html_e( 'Cart Behavior', 'dclg-24siteshop' ); ?></th>
						<td>
							<label>
								<input type="checkbox" name="<?php echo esc_attr( $option_name ); ?>[enforce_single]" value="1" <?php checked( 1, (int) $opts['enforce_single'] ); ?> />
								<?php esc_html_e( 'Disallow multiple products in cart (single-product cart).', 'dclg-24siteshop' ); ?>
							</label>
							<p class="description"><?php esc_html_e( 'When enabled, every new add-to-cart will replace the existing cart contents.', 'dclg-24siteshop' ); ?></p>
						</td>
					</tr>

					<tr>
						<th scope="row"><?php esc_html_e( 'Notices on Cart/Checkout', 'dclg-24siteshop' ); ?></th>
						<td>
							<label>
								<input type="checkbox" name="<?php echo esc_attr( $option_name ); ?>[suppress_notices]" value="1" <?php checked( 1, (int) $opts['suppress_notices'] ); ?> />
								<?php esc_html_e( 'Remove add/remove messages (and other notices) from Cart & Checkout pages.', 'dclg-24siteshop' ); ?>
							</label>
							<p class="description"><?php esc_html_e( 'Clears WooCommerce notices on cart and checkout, and hides the default “added to cart” message.', 'dclg-24siteshop' ); ?></p>
						</td>
					</tr>
				</table>

				<?php submit_button(); ?>
			</form>

			<h2 style="margin-top:2rem;"><?php esc_html_e( 'Generate Link', 'dclg-24siteshop' ); ?></h2>
			<p><?php esc_html_e( 'After saving the Product ID, click the button to generate your direct checkout link.', 'dclg-24siteshop' ); ?></p>
			<p>
				<button class="button button-primary" type="button" id="dclg_generate_btn"><?php esc_html_e( 'Generate Direct Checkout Link', 'dclg-24siteshop' ); ?></button>
			</p>
			<p>
				<input type="text" id="dclg_output_url" readonly class="large-text code" value="<?php echo esc_attr( $generated ); ?>" placeholder="<?php esc_attr_e( 'Your link will appear here...', 'dclg-24siteshop' ); ?>" onclick="this.select();document.execCommand('copy');" />
				<br/>
				<small><?php esc_html_e( 'Click the field to select and copy the URL.', 'dclg-24siteshop' ); ?></small>
			</p>

			<script>
			(function(){
				var btn = document.getElementById('dclg_generate_btn');
				if(!btn) return;
				btn.addEventListener('click', function(){
					var idField = document.getElementById('dclg_product_id');
					var outField = document.getElementById('dclg_output_url');
					if(!idField) return;
					var pid = (idField.value || '').trim();
					if(!pid){
						alert('<?php echo esc_js( __( 'Please enter a Product ID and click Save Changes first.', 'dclg-24siteshop' ) ); ?>');
						return;
					}
					var base = '<?php echo esc_js( esc_url_raw( wc_get_checkout_url() ) ); ?>';
					try {
						var url = new URL(base, window.location.origin);
						url.searchParams.set('add-to-cart', pid);
						outField.value = url.toString();
					} catch(e) {
						// Fallback if URL API fails
						outField.value = base + (base.indexOf('?') === -1 ? '?' : '&') + 'add-to-cart=' + encodeURIComponent(pid);
					}
				});
			})();
			</script>
		</div>
		<?php
	}

	/**
	 * If single-product cart is enabled, empty cart before adding a new item.
	 */
	public static function maybe_enforce_single_product_cart( $passed, $product_id, $quantity ) {
		$opts = self::get_options();
		if ( ! empty( $opts['enforce_single'] ) && function_exists( 'WC' ) && WC()->cart ) {
			if ( WC()->cart->get_cart_contents_count() > 0 ) {
				WC()->cart->empty_cart();
			}
		}
		return $passed;
	}

	/**
	 * Suppress add-to-cart messages and clear notices on cart/checkout when enabled.
	 */
	public static function maybe_suppress_add_to_cart_message( $message, $products ) {
		$opts = self::get_options();
		if ( ! empty( $opts['suppress_notices'] ) ) {
			return '';
		}
		return $message;
	}

	public static function maybe_clear_cart_checkout_notices() {
		$opts = self::get_options();
		if ( empty( $opts['suppress_notices'] ) ) return;

		if ( function_exists( 'is_cart' ) && is_cart() ) {
			wc_clear_notices();
			add_action( 'woocommerce_before_cart', 'wc_clear_notices', 0 );
			add_action( 'woocommerce_before_cart_table', 'wc_clear_notices', 0 );
		}
		if ( function_exists( 'is_checkout' ) && is_checkout() && ! is_order_received_page() ) {
			wc_clear_notices();
			add_action( 'woocommerce_before_checkout_form', 'wc_clear_notices', 0 );
		}
	}
}

DCLG_24SiteShop::init();
